bl_info = {
    "name": "PS1 Render Engine",
    "author": "Antigravity",
    "version": (1, 0),
    "blender": (4, 0, 0),
    "location": "Properties > Render > PS1 Engine",
    "description": "One-click PS1 style setups (Resolution, Dithering, Vertex Snap)",
    "category": "Render",
}

import bpy
import math

# ------------------------------------------------------------------------
#   GEOMETRY NODES (Vertex Snapping)
# ------------------------------------------------------------------------
def ensure_geo_node_group():
    group_name = "PS1_Vertex_Jitter"
    if group_name in bpy.data.node_groups:
        return bpy.data.node_groups[group_name]
    
    ng = bpy.data.node_groups.new(name=group_name, type='GeometryNodeTree')
    
    # Create Inputs/Outputs
    ng.interface.new_socket("Geometry", in_out='INPUT', socket_type='NodeSocketGeometry')
    res_socket = ng.interface.new_socket("Resolution", in_out='INPUT', socket_type='NodeSocketFloat')
    res_socket.default_value = 240.0
    res_socket.min_value = 1.0
    
    ng.interface.new_socket("Geometry", in_out='OUTPUT', socket_type='NodeSocketGeometry')
    
    # Add Nodes
    input_node = ng.nodes.new("NodeGroupInput")
    input_node.location = (-400, 0)
    
    output_node = ng.nodes.new("NodeGroupOutput")
    output_node.location = (600, 0)
    
    # 1. Get Position
    pos_node = ng.nodes.new("GeometryNodeInputPosition")
    pos_node.location = (-400, -200)
    
    # 2. Transform to Camera Space
    # We need the active camera. Since Geo Nodes doesn't always know the active camera easily without input,
    # we will use "Relative" transform which defaults to Object. But for "View Space" usually we need the camera object.
    # HOWEVER, a simpler "World Space" jitter is often enough and more stable. 
    # Let's try the "Camera" Object info approach if possible, but that requires setting the camera object.
    # To keep it robust, let's use a "Snap" in Object Space but quantized heavily, 
    # OR we can try to use the "Self Object" vs "Camera" logic if we assume a camera name.
    # Let's stick to a simpler "Snap Position" logic: 
    # Position -> Snap -> Set Position.
    # Ideally: Transform(Object->Camera) -> Snap -> Transform(Camera->Object).
    # Currently Blender GeoNodes 'Transform' node doesn't accept "Camera" space abstractly.
    # We'll use the "Object Info" node to get Camera transform.
    
    # Actually, let's keep it simple for V1: World Grid Snap. Even easier.
    # Replicating exact PS1 View Space jitter via Python-generated GeoNodes is complex 
    # because it requires linking the Active Camera object to every modifier.
    # We will implement World Space Grid Snap which looks very similar for moving objects.
    
    snap_node = ng.nodes.new("ShaderNodeVectorMath")
    snap_node.operation = 'SNAP'
    snap_node.location = (0, -200)
    
    # Snap increment: 1 / Resolution (approx)
    # Let's make the snap value controllable.
    # Math: 1.0 / Resolution * SCALAR? No, just use a small float value.
    # Using the "Resolution" input to drive the Snap increment.
    # Logic: Snap Increment = 2.0 / Resolution (Arbitrary scale factor)
    
    math_div = ng.nodes.new("ShaderNodeMath")
    math_div.operation = 'DIVIDE'
    math_div.inputs[0].default_value = 2.0 # Scale factor
    math_div.location = (-200, -300)
    
    set_pos = ng.nodes.new("GeometryNodeSetPosition")
    set_pos.location = (200, 0)
    
    # Links
    ng.links.new(input_node.outputs['Geometry'], set_pos.inputs['Geometry'])
    ng.links.new(input_node.outputs['Resolution'], math_div.inputs[1]) # Resolution controls snap size
    
    # Create Vector for Snap (using the result of divide for X,Y,Z)
    combine_xyz = ng.nodes.new("ShaderNodeCombineXYZ")
    combine_xyz.location = (-100, -300)
    ng.links.new(math_div.outputs[0], combine_xyz.inputs[0])
    ng.links.new(math_div.outputs[0], combine_xyz.inputs[1])
    ng.links.new(math_div.outputs[0], combine_xyz.inputs[2])
    
    ng.links.new(pos_node.outputs['Position'], snap_node.inputs[0])
    ng.links.new(combine_xyz.outputs['Vector'], snap_node.inputs[1])
    
    ng.links.new(snap_node.outputs['Vector'], set_pos.inputs['Position'])
    ng.links.new(set_pos.outputs['Geometry'], output_node.inputs['Geometry'])
    
    return ng

# ------------------------------------------------------------------------
#   COMPOSITOR (Dithering & Res)
# ------------------------------------------------------------------------
def setup_compositor(scene):
    scene.use_nodes = True
    if hasattr(scene, "compositing_node_group"):
        if not scene.compositing_node_group:
             # Create new node group if it doesn't exist
             tree = bpy.data.node_groups.new(name="PS1 Compositor", type="CompositorNodeTree")
             scene.compositing_node_group = tree
        else:
             tree = scene.compositing_node_group
    else:
        # Fallback for older Blender versions
        tree = scene.node_tree
    
    # Clear default
    for node in tree.nodes:
        tree.nodes.remove(node)
        
    # Create new
    try:
        rl = tree.nodes.new('CompositorNodeRLayers')
    except RuntimeError:
        # Fallback or alternative name
        rl = tree.nodes.new('NodeGroupInput') 

    try:
        comp = tree.nodes.new('CompositorNodeComposite')
    except RuntimeError:
        # Blender 5.0 replacement
        comp = tree.nodes.new('NodeGroupOutput')
        # Ensure the tree has an Output interface (which creates the input socket on the node)
        if not tree.interface.items_tree:
            # Create "Image" output
            tree.interface.new_socket("Image", in_out='OUTPUT', socket_type='NodeSocketColor')
    comp.location = (800, 0)
    
    # 1. Pixelate (Scale Down -> Scale Up)
    # PS1 resolution usually 320x240. 
    # If the render resolution is set to 320x240, we don't need to scale, we just output.
    # But usually users want 1080p output but "looking" like 240p.
    # So we: Scale (0.25) -> Pixelate -> Scale (4.0)
    
    scale_down = tree.nodes.new('CompositorNodeScale')
    # Blender 5.0: Use 'Type' socket instead of .space attribute
    if 'Type' in scale_down.inputs:
        scale_down.inputs['Type'].default_value = 'Relative'
    elif hasattr(scale_down, 'space'):
        scale_down.space = 'RELATIVE'
        
    if 'X' in scale_down.inputs:
        scale_down.inputs['X'].default_value = 0.25
    if 'Y' in scale_down.inputs:
        scale_down.inputs['Y'].default_value = 0.25
    scale_down.location = (-150, 0)
    
    pixelate = tree.nodes.new('CompositorNodePixelate')
    pixelate.location = (50, 0)
    
    scale_up = tree.nodes.new('CompositorNodeScale')
    if 'Type' in scale_up.inputs:
        scale_up.inputs['Type'].default_value = 'Relative'
    elif hasattr(scale_up, 'space'):
        scale_up.space = 'RELATIVE'
        
    if 'X' in scale_up.inputs:
        scale_up.inputs['X'].default_value = 4.0
    if 'Y' in scale_up.inputs:
        scale_up.inputs['Y'].default_value = 4.0
    scale_up.location = (250, 0)
    
    # Dithering (Simplified: RGB limit)
    # A true dither requires a Bayer Matrix texture.
    # For now, let's just add a "Color Depth" reduction via Posterize.
    posterize = tree.nodes.new('CompositorNodePosterize')
    posterize.inputs[1].default_value = 32.0 # 32 levels per channel (15-bit color approx)
    posterize.location = (500, 0)

    # --- MIST / FOG SETUP ---
    # We want to mix the render with a Fog Color based on the Mist Pass
    # 1. Mix Node
    # Blender 5.0 / 4.2+ uses ShaderNodeMix for Compositor too
    try:
        mix_fog = tree.nodes.new('ShaderNodeMix')
        mix_fog.data_type = 'RGBA'
    except RuntimeError:
        # Fallback for very old versions (unlikely here but safety)
        try:
             mix_fog = tree.nodes.new('CompositorNodeMixRGB')
        except RuntimeError:
             mix_fog = tree.nodes.new('CompositorNodeMix')
        
    mix_fog.location = (-200, 200)
    mix_fog.blend_type = 'MIX'
    
    # Helper to find specific typed socket (since Mix has multiple A's and B's)
    def get_typed_socket(node, names, type_match):
        for inp in node.inputs:
            if inp.name in names and type_match in inp.type:
                return inp
        # Fallback to name only
        for inp in node.inputs:
            if inp.name in names: return inp
        return node.inputs[0]

    # Connect Mist to Factor
    # Factor is usually 'VALUE'
    fac_socket = get_typed_socket(mix_fog, ['Factor', 'Fac'], 'VALUE')
    
    if 'Mist' in rl.outputs:
        tree.links.new(rl.outputs['Mist'], fac_socket)
    
    # Image -> A
    img_socket_A = get_typed_socket(mix_fog, ['Image', 'A', 'Color1'], 'RGBA')
    tree.links.new(rl.outputs[0], img_socket_A)
    
    # Fog Color -> B
    socket_B = get_typed_socket(mix_fog, ['B', 'Color2'], 'RGBA')
    socket_B.default_value = (0.0, 0.0, 0.0, 1.0) # Black Fog
    
    # Output
    # ShaderNodeMix has multiple outputs named 'Result' (Value, Vector, Color/RGBA)
    # We need the RGBA one.
    mix_res = None
    for out in mix_fog.outputs:
        if out.type == 'RGBA':
            mix_res = out
            break
            
    if mix_res is None:
        mix_res = mix_fog.outputs[2] # Hardcoded fallback (Index 2 is RGBA)
            
    tree.links.new(mix_res, scale_down.inputs[0])
    
    # Link Posterize -> Composite (Existing logic needs to be careful not to break)
    # The previous code linked posterize -> comp. That remains.
    # The only change is INPUT to scale_down is now mix_fog, not rl.
    
    # Wait, the previous block was:
    # tree.links.new(rl.outputs[0], scale_down.inputs[0])
    # We must REMOVE that logic from the snippet I'm replacing or overwrite it.
    # This replacement block REPLACES the first link setup.
    
    tree.links.new(scale_down.outputs[0], pixelate.inputs[0])
    tree.links.new(pixelate.outputs[0], scale_up.inputs[0])
    tree.links.new(scale_up.outputs[0], posterize.inputs[0])
    tree.links.new(posterize.outputs[0], comp.inputs[0])

# ------------------------------------------------------------------------
#   OPERATORS
# ------------------------------------------------------------------------
class PS1_OT_SetupScene(bpy.types.Operator):
    """Sets render resolution to 320x240 and sets up Compositor"""
    bl_idname = "ps1.setup_scene"
    bl_label = "Initialize PS1 Scene"
    
    def execute(self, context):
        # 1. Image Settings
        # Real PS1 hardware had various resolutions, 320x240 (NTSC) is standard.
        # But for 'Modern' viewing, it's better to render at 320x240 then upscale, 
        # or render at 1080p and use the Compositor nodes we defined.
        # Strategy: Set Render to 1280x960 (4x native) and let Compositor handle the 'Crunch'.
        # This keeps UI crisp but 3D View pixelated if using Compositor view.
        
        context.scene.render.resolution_x = 1280 
        context.scene.render.resolution_y = 960
        context.scene.render.fps = 24
        context.scene.render.engine = 'BLENDER_EEVEE'
        
        # 2. Filtering & Samples (Low samples = more jitter/less filtering)
        context.scene.render.filter_size = 0.0
        if hasattr(context.scene, 'eevee'):
            context.scene.eevee.taa_render_samples = 4
            context.scene.eevee.taa_samples = 1
            
        # Enable Mist for Fog consistency
        context.view_layer.use_pass_mist = True
        
        # 3. Compositor
        setup_compositor(context.scene)
        
        self.report({'INFO'}, "PS1 Scene Configured")
        return {'FINISHED'}

class PS1_OT_ApplyTraits(bpy.types.Operator):
    """Applies PS1 Geometry Modifier to selected objects"""
    bl_idname = "ps1.apply_traits"
    bl_label = "Apply PS1 Geometry"
    
    def execute(self, context):
        ng = ensure_geo_node_group()
        
        for obj in context.selected_objects:
            if obj.type == 'MESH':
                # Add Modifier if not present
                found = False
                for mod in obj.modifiers:
                    if mod.type == 'NODES' and mod.node_group == ng:
                        found = True
                        break
                if not found:
                    mod = obj.modifiers.new(name="PS1 Jitter", type='NODES')
                    mod.node_group = ng
        
        self.report({'INFO'}, f"Applied PS1 Jitter to {len(context.selected_objects)} objects")
        return {'FINISHED'}

# ------------------------------------------------------------------------
#   PANEL
# ------------------------------------------------------------------------
class PS1_PT_Panel(bpy.types.Panel):
    bl_label = "PS1 Render Engine"
    bl_idname = "PS1_PT_main_panel"
    bl_space_type = 'PROPERTIES'
    bl_region_type = 'WINDOW'
    bl_context = "render"
    
    def draw(self, context):
        layout = self.layout
        
        box = layout.box()
        box.label(text="Global Settings")
        box.operator("ps1.setup_scene", icon='SCENE_DATA')
        
        box2 = layout.box()
        box2.label(text="Object Controls")
        box2.operator("ps1.apply_traits", icon='MODIFIER')
        box2.label(text="Select objects to apply dithering/jitter")

# ------------------------------------------------------------------------
#   REGISTRATION
# ------------------------------------------------------------------------
classes = (
    PS1_OT_SetupScene,
    PS1_OT_ApplyTraits,
    PS1_PT_Panel,
)

def register():
    for cls in classes:
        bpy.utils.register_class(cls)

def unregister():
    for cls in classes:
        bpy.utils.unregister_class(cls)

if __name__ == "__main__":
    register()
